var Path = require('path');
var Promise = require('bluebird');
var FS = Promise.promisifyAll(require('fs-extra'));
var SessionManager = require('./SessionManager.js');

var Engine = function(goEmulate, binDir, 
    dataDir, portBase,
    webPort,healthPort, 
    upgradePort,publicDataPort,
    privateDataPort,udpRpcPort, 
    isServer, sensorIp,instanceIndex
) {
    this._goEmulate = goEmulate;
    this._rootDir = Path.join(binDir,'..','..');
    this._appDir =  Path.relative(this._rootDir,binDir);
    this._dataDir = dataDir;
    this._portBase = portBase;
    this._webPort = webPort;
    this._healthPort = healthPort;
    this._upgradePort = upgradePort;
    this._publicDataPort = publicDataPort;
    this._privateDataPort = privateDataPort;
    this._udpRpcPort = udpRpcPort;
    this._isServer = isServer;
    this._sensorIp = sensorIp;
    this._instanceIndex = instanceIndex;
};

Engine.prototype.load = function() {
    console.log('Engine: initializing data dir');
    this.initializeDataDir();
    return this.loadConfig().then(function(config) {
        console.log('Engine: config loaded');
        var config = this.mergeConfig(config);

        this._config = config;

       
        this.printConfig(config);
        
        if (this._use32Bit) this._appDir = "bin\\win32"; // use the 32 bit version of kframework by setting appDir to the 32 bit binaries directory

        this._goEmulate.setAppDir(Path.join(this._rootDir, this._appDir));
        this._goEmulate.setDataDir(Path.join(this._dataDir));
        this._version = this._goEmulate.getInfo()
        
        console.log('Engine: initializing objects');

        this.repository = new this._goEmulate.Repository(Path.join(this._dataDir, "datasets"));
        this.sessions = new SessionManager(this._goEmulate, this._isServer,
            this.repository, {
                maxSessionCount: config.maxSessionCount,
                connectionTimeout: config.connectionTimeout,
                activityTimeout: config.activityTimeout,
                maxIpSessionCount: config.maxIpSessionCount,
                portBase: this._portBase,
                webPort: this._webPort,
                healthPort: this._healthPort,
                upgradePort: this._upgradePort,
                publicDataPort: this._publicDataPort,
                privateDataPort: this._privateDataPort,
                udpRpcPort: this._udpRpcPort,
                ipAddress: this._sensorIp,
                instanceIndex: this._instanceIndex
            });
        this.printParams();
        
        console.log('Engine: initializing session dir');
        if (this._isServer){
            FS.ensureDir(Path.join(this._dataDir,'sessions'));
        }
        
        console.log('Engine: setting session dir');
        return this.setSessionLogDir();
    }.bind(this));
};

Engine.prototype.initializeDataDir = function() {
    if (this._dataDir != Path.join(this._rootDir, 'data','GoEmulate')){

        //makes sure the dirs exist.
        FS.ensureDir(this._dataDir)
    }
}

Engine.prototype.setSessionLogDir = function() {
    var sessionLogDir = Path.join(this._dataDir, 'sessions');

    return FS.statAsync(sessionLogDir).then(function(result) {
        if (!result.isDirectory()) {
            throw 'Not a directory';
        }

        console.log('Logging sessions to ' + sessionLogDir);
        this.sessions.setLogDir(sessionLogDir);
    }.bind(this)).catch(function(err) {
        console.log('Warning: logging disabled. Cannot use ' + sessionLogDir);
    });
};

Engine.prototype.loadConfig = function() {

    var configName = this.getConfigName();
    var configPath = Path.join(this._rootDir, 'data', 'GoEmulate', configName);

    return FS.readFileAsync(configPath).then(function(result) {
        var config = JSON.parse(result);
        console.log('Read config file at ' + configPath);
        return config;
    }).catch(function(err) {
        console.log('Warning: cannot read config at ' + configPath);
    });
};

Engine.prototype.mergeConfig = function(override) {
    var defaults = {
        maxSessionCount: 10,
        maxIpSessionCount: 5,
        connectionTimeout: 10,
        activityTimeout: 1800,
        serverPort: 0,      //auto
        redirectUrl: null   //auto
    };

    for (prop in override) {
        defaults[prop] = override[prop];
    }

    return defaults;
};

Engine.prototype.printParams = function() {
    console.log('Parameters: ');

    console.log('  portBase: ' + this._portBase);
    console.log('  webPort: ' + this._webPort);
    console.log('  healthPort: ' + this._healthPort);
    console.log('  privateDataPort: ' + this._privateDataPort);
    console.log('  publicDataPort: ' + this._publicDataPort);
    console.log('  upgradePort: ' + this._upgradePort);
    console.log('  udpRpcPort: ' + this._udpRpcPort);
    console.log('  isServer: ' + this._isServer);
    console.log('  instanceIndex: ' + this._instanceIndex);
};

Engine.prototype.printConfig = function(config) {
    console.log('Configuration: ');

    for (prop in config) {
        console.log('  ' + prop + ': ' + config[prop]);
    }
};

Engine.prototype.getConfigName = function() {
    
    var configFile = this._isServer ? 'config.json' : 'config-desktop.json';
    console.log('  configFile: ' + configFile);
    return configFile;
};

Engine.prototype.getServerPort = function() {
    return this._config.serverPort;
};

Engine.prototype.getRedirectUrl = function() {
    return this._config.redirectUrl;
};

Engine.prototype.getIsServer = function() {
    return this._isServer;
};

Engine.prototype.getInfo = function() {
    return this._version;
};

module.exports = Engine;
